/**
 * Country and Language Manager
 * Integrates country-json for comprehensive country/language data
 */

const countryByLanguages = require('country-json/src/country-by-languages.json');
const countryByCallingCode = require('country-json/src/country-by-calling-code.json');
const countryByAbbreviation = require('country-json/src/country-by-abbreviation.json');

class CountryLanguageManager {
  constructor() {
    this.countries = countryByAbbreviation;
    this.callingCodes = countryByCallingCode;
    this.countryLanguages = countryByLanguages;
    
    // Build language index
    this.languageIndex = this.buildLanguageIndex();
    
    // Most common languages for quick selection
    this.popularLanguages = [
      { code: 'en', name: 'English', countries: 'USA, UK, Australia, Canada' },
      { code: 'pt', name: 'Português', countries: 'Brazil, Portugal, Angola' },
      { code: 'es', name: 'Español', countries: 'Spain, Mexico, Argentina' },
      { code: 'fr', name: 'Français', countries: 'France, Canada, Belgium' },
      { code: 'de', name: 'Deutsch', countries: 'Germany, Austria, Switzerland' },
      { code: 'it', name: 'Italiano', countries: 'Italy, Switzerland' },
      { code: 'zh', name: '中文', countries: 'China, Taiwan, Singapore' },
      { code: 'ja', name: '日本語', countries: 'Japan' },
      { code: 'ko', name: '한국어', countries: 'South Korea' },
      { code: 'ar', name: 'العربية', countries: 'Saudi Arabia, UAE, Egypt' },
      { code: 'ru', name: 'Русский', countries: 'Russia, Kazakhstan' },
      { code: 'hi', name: 'हिन्दी', countries: 'India' }
    ];
  }

  /**
   * Build index of all languages with their countries
   */
  buildLanguageIndex() {
    const langIndex = new Map();
    
    for (const country of this.countryLanguages) {
      if (!country.languages) continue;
      
      let langs = [];
      
      // Handle different data types
      if (typeof country.languages === 'string') {
        if (country.languages === '') continue;
        langs = country.languages.split(',').map(l => l.trim().toLowerCase());
      } else if (Array.isArray(country.languages)) {
        langs = country.languages.map(l => String(l).trim().toLowerCase());
      } else {
        continue; // Skip if not string or array
      }
      
      for (const lang of langs) {
        if (!lang) continue; // Skip empty strings
        if (!langIndex.has(lang)) {
          langIndex.set(lang, []);
        }
        langIndex.get(lang).push(country.country);
      }
    }
    
    return langIndex;
  }

  /**
   * Get country by calling code
   */
  getCountryByCallingCode(code) {
    // Remove + and non-digits
    const cleanCode = code.replace(/\D/g, '');
    
    for (const country of this.callingCodes) {
      if (country.calling_code === `+${cleanCode}` || 
          country.calling_code === cleanCode) {
        return country.country;
      }
    }
    
    return null;
  }

  /**
   * Get country by phone number
   */
  getCountryByPhone(phoneNumber) {
    const cleanPhone = phoneNumber.replace(/\D/g, '');
    
    // Try common calling codes by length
    const codes = [
      cleanPhone.substring(0, 1),  // 1 digit (1 for US/Canada)
      cleanPhone.substring(0, 2),  // 2 digits (55 for Brazil, 44 for UK)
      cleanPhone.substring(0, 3),  // 3 digits (258 for Mozambique)
      cleanPhone.substring(0, 4)   // 4 digits
    ];
    
    for (const code of codes) {
      const country = this.getCountryByCallingCode(code);
      if (country) {
        return { country, callingCode: code };
      }
    }
    
    return null;
  }

  /**
   * Detect likely language from phone number
   */
  detectLanguageFromPhone(phoneNumber) {
    const countryInfo = this.getCountryByPhone(phoneNumber);
    
    if (!countryInfo) {
      return 'en'; // Default to English
    }
    
    // Find country in language data
    const countryData = this.countryLanguages.find(
      c => c.country.toLowerCase() === countryInfo.country.toLowerCase()
    );
    
    if (!countryData || !countryData.languages) {
      return 'en';
    }
    
    // Get first language (usually primary)
    let primaryLang = '';
    
    if (typeof countryData.languages === 'string') {
      primaryLang = countryData.languages.split(',')[0].trim().toLowerCase();
    } else if (Array.isArray(countryData.languages) && countryData.languages.length > 0) {
      primaryLang = String(countryData.languages[0]).trim().toLowerCase();
    } else {
      return 'en';
    }
    
    // Map full language names to ISO codes
    const langMap = {
      'english': 'en',
      'portuguese': 'pt',
      'spanish': 'es',
      'french': 'fr',
      'german': 'de',
      'italian': 'it',
      'chinese': 'zh',
      'japanese': 'ja',
      'korean': 'ko',
      'arabic': 'ar',
      'russian': 'ru',
      'hindi': 'hi',
      'dutch': 'nl',
      'swedish': 'sv',
      'norwegian': 'no',
      'danish': 'da',
      'finnish': 'fi',
      'polish': 'pl',
      'turkish': 'tr',
      'thai': 'th',
      'vietnamese': 'vi',
      'indonesian': 'id',
      'malay': 'ms',
      'tagalog': 'tl'
    };
    
    return langMap[primaryLang] || 'en';
  }

  /**
   * Get popular languages formatted for selection
   */
  getPopularLanguagesFormatted() {
    let message = '🌍 *Select Language:*\n\n';
    message += '━━━━━━━━━━━━━━━━━━━━\n';
    message += '📌 *Most Common:*\n\n';
    
    this.popularLanguages.forEach((lang, index) => {
      message += `${index + 1}️⃣ *${lang.name}* (${lang.code})\n`;
      message += `   ${lang.countries}\n\n`;
    });
    
    message += '━━━━━━━━━━━━━━━━━━━━\n';
    message += '💡 Type the *number* or *language code*\n';
    message += 'Example: 1 or en or pt\n\n';
    message += '🔍 Need another language? Type the ISO code\n';
    message += 'Examples: nl, sv, no, da, fi, pl, tr, th, vi, id, ms, tl';
    
    return message;
  }

  /**
   * Parse language selection
   */
  parseLanguageSelection(input) {
    const clean = input.trim().toLowerCase();
    
    // Check if it's a number (1-12)
    const num = parseInt(clean);
    if (!isNaN(num) && num >= 1 && num <= this.popularLanguages.length) {
      return this.popularLanguages[num - 1].code;
    }
    
    // Check if it's a language code
    if (/^[a-z]{2,3}$/.test(clean)) {
      return clean;
    }
    
    return null;
  }

  /**
   * Get language name by code
   */
  getLanguageName(code) {
    const langNames = {
      'en': 'English',
      'pt': 'Português',
      'es': 'Español',
      'fr': 'Français',
      'de': 'Deutsch',
      'it': 'Italiano',
      'zh': '中文',
      'ja': '日本語',
      'ko': '한국어',
      'ar': 'العربية',
      'ru': 'Русский',
      'hi': 'हिन्दी',
      'nl': 'Nederlands',
      'sv': 'Svenska',
      'no': 'Norsk',
      'da': 'Dansk',
      'fi': 'Suomi',
      'pl': 'Polski',
      'tr': 'Türkçe',
      'th': 'ไทย',
      'vi': 'Tiếng Việt',
      'id': 'Bahasa Indonesia',
      'ms': 'Bahasa Melayu',
      'tl': 'Tagalog'
    };
    
    return langNames[code] || code.toUpperCase();
  }

  /**
   * Validate language code
   */
  isValidLanguageCode(code) {
    if (!code || typeof code !== 'string') {
      return false;
    }
    
    const clean = code.toLowerCase().trim();
    
    // Must be 2-3 lowercase letters
    return /^[a-z]{2,3}$/.test(clean);
  }

  /**
   * Get all supported languages
   */
  getAllLanguages() {
    const allLangs = new Set();
    
    for (const country of this.countryLanguages) {
      if (!country.languages) continue;
      
      let langs = [];
      if (typeof country.languages === 'string') {
        langs = country.languages.split(',').map(l => l.trim().toLowerCase());
      } else if (Array.isArray(country.languages)) {
        langs = country.languages.map(l => String(l).trim().toLowerCase());
      }
      
      langs.forEach(lang => {
        if (lang) allLangs.add(lang);
      });
    }
    
    return Array.from(allLangs).sort();
  }
}

module.exports = new CountryLanguageManager();
